/* $Id: multipass2.c,v 1.5 1999/01/30 00:34:18 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Do a few checks that the pass numbers from multipass data are
   correct (the pass bits in the 32_SERV data format).  Optionally
   print out the pass numbers, so we can see if they look right. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	BLOCKSIZE		512
#define	NSCAN			20
#define	NMOD_MAX		4
#define	NCHAN_MAX		(NMOD_MAX * E1432_INPUT_CHANS)
#define	E1432_PASS_MAX		8
#define	E1433_PASS_MAX		17
#define	E1432_204_PASS_MAX	13
#define	P_MAX			E1432_204_PASS_MAX

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: multipass2.c,v 1.5 1999/01/30 00:34:18 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      double span, int over)
{
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32_SERV));
    CHECK(e1432_set_decimation_output(hw, group, E1432_MULTIPASS));
    if (over)
	CHECK(e1432_set_decimation_oversample(hw, group,
					      E1432_DECIMATION_OVERSAMPLE_ON));
    CHECK(e1432_set_span(hw, group, span));
    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

static int
examine_pass(LONGSIZ32 *buf, double span, int over, int first, int verbose)
{
    static LONGSIZ32 save_buf[BLOCKSIZE];
    double  top_span;
    long    i, tmp;
    int     top, bottom, found_bottom, print_modulus;

    if ((int) ((int) (4000 / span + 0.5) * span + 0.5) == 4000)
	top_span = 4000;
    else
	top_span = 20000;
    top = 0;
    while (span * 1.00001 < top_span)
    {
	span *= 2;
	top++;
    }
    bottom = top;
    tmp = 1;
    while (tmp < BLOCKSIZE)
    {
	tmp *= 2;
	bottom++;
    }
    if (!over)
	/* Non-oversampled might not have the single bottom pass
	   sample, depending on sequence order. */
	bottom--;
    if (bottom > P_MAX)
	bottom = P_MAX;
    if (top >= 10)
	print_modulus = 16;
    else
	print_modulus = 32;

    /* Maybe print out the pass numbers, to make debugging easier */
    if ((first && verbose > 2) || verbose > 3)
    {
	for (i = 0; i < BLOCKSIZE; i++)
	{
	    (void) printf("%d ", buf[i] & 0xf);
	    if (i % print_modulus == print_modulus - 1)
		(void) printf("\n");
	}
	if (i % print_modulus != 0)
	    (void) printf("\n");
    }

    /* Do a few crude checks on the pass numbers */
    if ((buf[0] & 0xf) != top && (buf[1] & 0xf) != top)
    {
	/* When not oversampled, it make take one more sample to see
	   the top pass */
	if (over || (buf[2] & 0xf) != top)
	{
	    (void) fprintf(stderr,
			   "%s: top pass %d not found "
			   "in first three samples: %ld, %ld, %ld\n",
			   progname, top, buf[0] & 0xf, buf[1] & 0xf,
			   buf[2] & 0xf);
	    return -1;
	}
    }
    if ((buf[0] & 0xf) != top + 1 && (buf[1] & 0xf) != top + 1 &&
	(buf[2] & 0xf) != top + 1 && (buf[3] & 0xf) != top + 1)
    {
	/* When not oversampled, it make take one more sample to see
	   the top + 1 pass */
	if (over || (buf[4] & 0xf) != top + 1)
	{
	    (void) fprintf(stderr,
			   "%s: top + 1 pass %d not found "
			   "in first five samples: %ld, %ld, %ld, %ld, %ld\n",
			   progname, top + 1, buf[0] & 0xf, buf[1] & 0xf,
			   buf[2] & 0xf, buf[3] & 0xf, buf[4] & 0xf);
	    return -1;
	}
    }
    found_bottom = 0;
    for (i = 0; i < BLOCKSIZE; i++)
    {
	if ((buf[i] & 0xf) >= bottom)
	    found_bottom = 1;
	/* Allow a pass number of 15 to work also.  This is what the
	   Vibrato code uses for the dummy pass.  I think Sonata uses
	   "1", which equals the maximum valid pass, which doesn't
	   make sense, so I'm not currently allowing for it. */
	if (((buf[i] & 0xf) < top ||
	     (buf[i] & 0xf) > P_MAX) &&
	    (buf[i] & 0xf) != 15)
	{
	    (void) fprintf(stderr,
			   "%s: out of range pass value at %ld: %ld\n",
			   progname, i, buf[i] & 0xf);
	    return -1;
	}
    }
    if (!found_bottom)
    {
	(void) fprintf(stderr,
		       "%s: never found bottom pass %ld\n",
		       progname, bottom);
	return -1;
    }

    if (first)
	/* Save first channel's data */
	(void) memcpy(save_buf, buf, sizeof save_buf);
    else
	/* All subsequent channels should have the same pass numbers */
	for (i = 0; i < BLOCKSIZE; i++)
	    if ((buf[i] & 0xf) != (save_buf[i] & 0xf))
	    {
		(void) fprintf(stderr,
			       "%s: mismatch at %ld, got %ld, expected %ld\n",
			       progname, i, buf[i] & 0xf,
			       save_buf[i] & 0xf);
		return -1;
	    }

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    double span, int over, int verbose)
{
    LONGSIZ32 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     scan, chan;

    CHECK(e1432_init_measure(hw, group));

    for (scan = 0; scan < NSCAN; scan++)
    {
	/* Wait for block available */
	CHECK(wait_block_avail(hw, group, scan, verbose,
			       BLOCKSIZE, span));

	/* Read the data */
	for (chan = 0; chan < nchan; chan++)
	{
	    if (verbose > 1)
		(void) printf("Reading chan %d\n", chan);

	    CHECK(e1432_read_raw_data(hw, chan_list[chan],
				      E1432_TIME_DATA,
				      buffer, BLOCKSIZE,
				      &trailer, &count));
	    if (count != BLOCKSIZE)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_read_float64_data: "
			       "actual count was %ld\n",
			       progname, count);
		return -1;
	    }

	    CHECK(check_trailer(&trailer, 51200, span,
				chan_list[chan], 0));

	    CHECK(examine_pass(buffer, span, over, chan == 0, verbose));
	}
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
	   "Usage: %s [-ouvV] [-L laddr] [-n nchan] [-N nmod] [-s span]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-o: Use oversampled data\n"
		   "\t-s: Use <span>, default 5000\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  span;
    char   *p;
    int     c, i, nmod, over, verbose;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    over = 0;
    span = 5000;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:os:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'o':
	    over = 1;
	    break;
	case 's':
	    if (sscanf(optarg, "%lg", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, span, over) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, span, over, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
